<?php

namespace Drupal\secure_privacy\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Cache\Cache;

/**
 * Configures Secure Privacy settings.
 */
class SecurePrivacySettingsForm extends ConfigFormBase {

  /**
   * Constructs a \Drupal\secure_privacy\Form\SecurePrivacySettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager
  ) {
    parent::__construct($config_factory, $typed_config_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'secure_privacy_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['secure_privacy.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('secure_privacy.settings');

    $form['script_content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Secure Privacy Script'),
      '#default_value' => $config->get('script_content'),
      '#description' => $this->t('Enter your Secure Privacy script code. The script will be automatically injected at the beginning of the &lt;head&gt; tag of your website.'),
      '#required' => TRUE,
      '#rows' => 5,
      '#attributes' => [
        'placeholder' => 'Enter your Secure Privacy Installation script...',
      ],
      '#prefix' => '<div class="secure-privacy-script-wrapper">',
      '#suffix' => '<div class="description">' . 
        $this->t('Note: The script will be placed as early as possible in the &lt;head&gt; section for optimal functionality.') . 
        '</div></div>',
    ];

    $form['is_active'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Secure Privacy'),
      '#default_value' => $config->get('is_active'),
      '#description' => $this->t('Enable or disable the Secure Privacy script. When disabled, the script will be immediately removed from your website.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $script_content = $form_state->getValue('script_content');
    
    if (!empty($script_content)) {
      // Basic validation to ensure it contains a script tag
      if (!preg_match('/<script[^>]*>[^<]*(?:<(?!\/script>)[^<]*)*<\/script>/i', $script_content)) {
        $form_state->setErrorByName('script_content', $this->t('Please enter a valid script tag. The content should include both opening and closing script tags.'));
        return;
      }

      // Check if it's a Secure Privacy script
      if (stripos($script_content, 'secureprivacy.ai') === FALSE) {
        $form_state->setErrorByName('script_content', $this->t('The script does not appear to be from Secure Privacy. Please verify the script content.'));
        return;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    try {
      $old_config = $this->config('secure_privacy.settings');
      $was_active = $old_config->get('is_active');
      $is_active = $form_state->getValue('is_active');

      // Save the new configuration
      $this->config('secure_privacy.settings')
        ->set('script_content', $form_state->getValue('script_content'))
        ->set('is_active', $is_active)
        ->save();

      // Clear page caches to ensure script state is updated
      Cache::invalidateTags(['rendered']);

      // Set appropriate message based on state change
      if ($was_active && !$is_active) {
        $this->messenger()->addStatus($this->t('Secure Privacy has been disabled. The script has been removed from your website.'));
        \Drupal::logger('secure_privacy')->notice('Script disabled and removed from website');
      } elseif (!$was_active && $is_active) {
        $this->messenger()->addStatus($this->t('Secure Privacy has been enabled. The script will now be loaded on your website.'));
        \Drupal::logger('secure_privacy')->notice('Script enabled and will be injected');
      } else {
        $this->messenger()->addStatus($this->t('The configuration options have been saved.'));
      }

      parent::submitForm($form, $form_state);
    }
    catch (\Exception $e) {
      \Drupal::logger('secure_privacy')->error('Error saving settings: @error', ['@error' => $e->getMessage()]);
      $this->messenger()->addError($this->t('Failed to save settings. Please check the logs for details.'));
      throw $e;
    }
  }
}

